/* class IntegralImage
* Implementation of generalized integral image
*
* Author: Mikhail Sizintsev
* Date: 2007-06-04
*/

#ifndef __INTEGRAL_IMAGE_H__
#define __INTEGRAL_IMAGE_H__

#define INTEGRAL_IMAGE_MAX_ORDER 10

#include "Image.h"

class IntegralImage {
protected:
	// image dimensions
	int width;
	int height;
	// order of the integral image
	int order;

	// kernel coefficients for extracting a value
	iType* coefs;
	iType kernelStep;

	// data pointer
	iType* data;
	// for correct zero-padding
	iType* lastVerticalStrip;
	iType* lastHorizontalStrip;
	iType lastElement;

	// original image
	Image* im;
	
	/// auxiliary inline methods
	
	
	// read .pgm image
	iType* __readPgmImage(char* filename, int* outWidth = NULL, int* outHeight = NULL);
	

public:
	/// default constructor
	IntegralImage();
	/// construct integral image by loading the image file (pgm supported)
	IntegralImage(char* filename, int _order = 0);
	/// construct integral image by processing the conventional image
	//IntegralImage(int _order, uImage<T>){}

	/// destructor
	virtual ~IntegralImage();

	// calculate the period of spline from its standard deviation
	int calcT(FLOAT sigma);

	// extracting aggregated values (without boundary check)
	inline iType at(int x, int y, int T, int normFactor = 1){
		iType result = 0;
		int dx, dy;
		for (int j = 0; j < order+1; j++){
			for (int i = 0; i < order+1; i++){
				dx = (int)(x+(order/2.0-i)*T); dy = (int)(y+(order/2.0-j)*T);				
				result += coefs[i]*coefs[j]*data[dy * width + dx];
			}
		}
		return result/normFactor;
	}


	//
	inline iType atDirect(int x, int y){ return data[y*width+x]; }

	// maximum and minimum of 2 numbers
	inline int __IImax2(int a, int b){ if (a > b) return a; else return b;}
	inline int __IImin2(int a, int b){ if (a < b) return a; else return b;}

	/* method for integral image of known order (1, 2 or 3)
	* very fast but no checks
	* USE WITH CAUTION!
	*/

	// order 1 accesor method
	inline iType at1(int x, int y, int T){
		return data[(y+T/2)*width+(x+T/2)] + data[(y-T/2)*width+(x-T/2)]
			- data[(y-T/2)*width+(x+T/2)] - data[(y+T/2)*width+(x-T/2)];
		
	}

	// order 2 accessor method
	inline iType at2(int x, int y, int T){
		return data[(y+T)*width + (x+T)] 
			- ((data[(y+T)*width + (x)])<<1)
			+ data[(y+T)*width + (x-T)]

			- ((data[(y)*width + (x+T)])<<1)
			+ ((data[(y)*width + (x)])<<2)
			- ((data[(y)*width + (x-T)])<<1)

			+ data[(y-T)*width + (x+T)] 
			- ((data[(y-T)*width + (x)])<<1)
			+ data[(y-T)*width + (x-T)];
	}


	// accessor methods
	inline int getWidth() { return width; }
	inline int getHeight() { return height; }
	inline int getOrder() { return order; }

	// dump coefs kernel; for testing purposes
	void dumpKernel(FILE* fp);

};

#endif
