#include "IntegralSift.h"

/// default constructor
IntegralSift::IntegralSift(){
	order = 0; maxT = 0; dT = 0;
	pSS = NULL; pIm = NULL;
	descriptorDim = FEATURE_DIMENSION;
}

/// initialize by passing the image
/* params:	im -- integral image
*			maxT -- maximum scale (in T)
*			dT -- step size for scale-space construction (in T)
*/
IntegralSift::IntegralSift(IntegralImage* im, int _maxT, int _dT){
	descriptorDim = FEATURE_DIMENSION;
	maxT = _maxT;
	dT = _dT;
	pIm = im;
	order = pIm->getOrder();
	pSS = new ScaleSpace(im->getWidth(), im->getHeight(), maxT/dT);
	pSS->clearBuffer();
	features.clear();
	features.reserve(MAX_NUMBER_OF_FEATURES);

	contrastThreshold = (FLOAT)3.76; //0;
	hessianThreshold = 10;
}

/// destructor
IntegralSift::~IntegralSift(){
	if (pSS != NULL) delete pSS; pSS = NULL;
}

/// construct scale-space layer-by-layer
void IntegralSift::constructScaleSpace(){
	int curT, margin;
	FLOAT *pCur, *pNext; // pointers to data buffer for faster access
//	iType normFactor;
	FLOAT normFactor;

	// create approximation go Gaussian scalespace
	for (int d = 0; d < pSS->getDepth(); d++){
		curT = calcT(d);
		margin = calcMargin(curT)+1;
		normFactor = (FLOAT)calcNormFactor(curT);
//		normFactor = calcNormFactor(curT);
//		printf("T = %d, order = %d, normFactor = %lld\n", curT, order, normFactor);
		switch(pIm->getOrder()){
			case 1:
				for (int y = margin; y < pSS->getHeight()-margin; y++){
					pCur = pSS->getData() + d*pSS->getHeight()*pSS->getWidth() + y*pSS->getWidth() + margin;	
					for (int x = margin; x < pSS->getWidth()-margin; x++){
//						pSS->set((FLOAT)((pIm->at1(x, y, curT))/normFactor), x, y, d);
						*pCur++ = (FLOAT)((pIm->at1(x, y, curT))/normFactor);
					}
				}
				break;
			case 2:
				for (int y = margin; y < pSS->getHeight()-margin; y++){
					pCur = pSS->getData() + d*pSS->getHeight()*pSS->getWidth() + y*pSS->getWidth() + margin;	
					for (int x = margin; x < pSS->getWidth()-margin; x++){
//						pSS->set((FLOAT)((pIm->at2(x, y, curT))/normFactor), x, y, d);
						*pCur++ = (FLOAT)(pIm->at2(x, y, curT)/normFactor);
					}
				}
				break;
			case 3:
				//break;
			case 4:
				//break;
			default:
				for (int y = margin; y < pSS->getHeight()-margin; y++){
					pCur = pSS->getData() + d*pSS->getHeight()*pSS->getWidth() + y*pSS->getWidth() + margin;	
					for (int x = margin; x < pSS->getWidth()-margin; x++){
//						pSS->set((FLOAT)((pIm->at(x, y, curT))/normFactor), x, y, d);
						*pCur++ = (FLOAT)(pIm->at(x, y, curT)/normFactor);
					}
				}		
		}
		
	}

	// create approxaimtion to laplacian scalspace (from gaussian scale-space by pairwise subtraction of layers)
	FLOAT ZERO = 0;
	for (int d = 0; d < pSS->getDepth()-1; d++){
		margin = calcMargin(calcT(d))+3;
		// make boundaries zero
		for (int y = 0; y < margin; y++){
			pCur = pSS->getData() + d*pSS->getHeight()*pSS->getWidth() + y*pSS->getWidth();	
			pNext = pSS->getData() + d*pSS->getHeight()*pSS->getWidth() + (pSS->getHeight()-1 - y)*pSS->getWidth();	
			for (int x = 0; x < pSS->getWidth(); x++){
				*pCur++ = ZERO;
				*pNext++ = ZERO;
			}
		}
		for (int y = margin; y < pSS->getHeight()-margin; y++){
			pCur = pSS->getData() + d*pSS->getHeight()*pSS->getWidth() + y*pSS->getWidth();	
			pNext = pSS->getData() + d*pSS->getHeight()*pSS->getWidth() + y*pSS->getWidth() + (pSS->getWidth()-margin);
			for (int x = 0; x < margin; x++){
				*pCur++ = ZERO;
				*pNext++ = ZERO;
			}
		}
		
		
		// process the rest of points
		for (int y = margin; y < pSS->getHeight()-margin; y++){
			pCur = pSS->getData() + d*pSS->getHeight()*pSS->getWidth() + y*pSS->getWidth() + margin;	
			pNext = pSS->getData() + (d+1)*pSS->getHeight()*pSS->getWidth() + y*pSS->getWidth() + margin;	
			for (int x = margin; x < pSS->getWidth()-margin; x++){
//				pSS->set(pSS->at(x, y, d) - pSS->at(x, y, d+1), x, y, d);
				*pCur = *pCur-*pNext; pCur++; pNext++;
			}
		}

	}
	// make the last layer uniformly zero
	for (int y = 0; y < pSS->getHeight(); y++){
		pCur = pSS->getData() + (pSS->getDepth()-1)*pSS->getHeight()*pSS->getWidth() + y*pSS->getWidth();	
		for (int x = 0; x < pSS->getWidth(); x++){
			*pCur++ = ZERO;
		}
	}

}

/// find sift features
void IntegralSift::findSift(){
	int margin, curT, window;
	bool isMax;
	FLOAT cur, curOr = 0;
	
	// for each point in the scale-space find if its a local maximum (scale-invariant version of 3x3)and satisfies validity conditions
	for (int d = 1; d < pSS->getDepth()-1; d++){
		curT = calcT(d);
		margin = calcMargin(curT) + 1;
		//window = pIm->__IImax2((int)pow(2.0,(double)ceil(log(calcSigma(curT)))), 1);
		window = pIm->__IImax2((int)calcSigma(curT), 1);
		printf("curT is %d, window is %d\n", curT, window);
		for (int y = margin+window; y < pSS->getHeight()-margin-window; y+=window){//y++){
			for (int x = margin+window; x < pSS->getWidth()-margin-window; x+=window){//x++){
				if (isValid(x, y, d)){
					isMax = true;
					cur = abs(pSS->at(x, y, d));
					for (int dy = -window; dy <= window; dy+=window){//dy++){
						for (int dx = -window; dx <= window; dx+=window){//dx++){
							if (dx == 0 && dy == 0){
								if (!(cur > abs(pSS->at(x,y,d-1)) && cur > abs(pSS->at(x,y,d+1)))){
									isMax = false; break;
								}
							}else if(!(cur > abs(pSS->at(x+dx, y+dy, d-1)) && 
								cur > abs(pSS->at(x+dx, y+dy, d)) &&
								cur > abs(pSS->at(x+dx, y+dy, d+1)))){
									isMax = false; break;
							}
							/*if (!(dx == 0 && dy == 0) && !(cur > abs(pSS->at(x+dx, y+dy, d-1)) && 
								cur > abs(pSS->at(x+dx, y+dy, d)) &&
								cur > abs(pSS->at(x+dx, y+dy, d+1)))){
								isMax = false;
								break;
							}*/								
						}
						if (!isMax) break;
					}
					if (isMax){ // record a new sift feature (maybe need to compute subpixel location here)
						//features.push_back(SiftFeature((FLOAT)x, (FLOAT)y, calcSigma(curT), curOr));						
						features.push_back(SiftFeature((FLOAT)y, (FLOAT)x, calcSigma(curT), curOr));						
					}
				}
			}
		}
	}
}

/// checks validity of a point to be a Sift feature (must pass the contrast + hessian test)
bool IntegralSift::isValid(int x, int y, int layer){
	if (x >= 1 && x < pSS->getWidth()-1 && y >=1 && y < pSS->getHeight()-1){
		if (abs(pSS->at(x, y, layer)) > contrastThreshold){
			FLOAT dxx, dyy, dxy;
			dxx = pSS->at(x-1, y, layer) - (FLOAT)2.0*pSS->at(x, y, layer) + pSS->at(x+1, y, layer);
			dyy = pSS->at(x, y-1, layer) - (FLOAT)2.0*pSS->at(x, y, layer) + pSS->at(x, y+1, layer);
			dxy = pSS->at(x, y, layer) + pSS->at(x+1,y+1, layer) - pSS->at(x+1, y, layer) - pSS->at(x, y+1, layer);
			if (mySquare(dxx + dyy)/(dxx*dyy - mySquare(dxy) + MY_EPS) <= mySquare(hessianThreshold + 1)/hessianThreshold)
				return true;
			else
				return false;
		}else
			return false;
	}else
		return false;	
}


/// dumps the computed sift feature in the temporary file
void IntegralSift::dumpFeatures(char* filename){
	FILE* ofp;
	ofp = fopen(filename, "wb");
	if (!ofp) {
		printf("Error: Unable to open file %s.\n\n", filename);
		exit(1);
	}

	// write the header
	fprintf( ofp, "%d %d\n", features.size(), descriptorDim);
   
	//write feature points
	SiftFeature *f;
	for (int i = 0; i < (int)features.size(); i++){
		f = &(features[i]);
		fprintf( ofp, "%f %f %f %f\n", f->x, f->y, f->scale, f->orientation);
		for (int j = 0; j < descriptorDim; j++){
			fprintf( ofp, "%d ", f->descriptor[j]);
		}
		fprintf( ofp, "\n");
	}

	fclose(ofp);	
}

void IntegralSift::compute(){
	constructScaleSpace();
	findSift();
	printf("%d SIFT features have been detected\n", features.size());
	dumpFeatures(OUT_FILENAME);
}
