/* class Image
* Implementation of SIFT using integral images
*
* Author: Mikhail Sizintsev
* Date: 2007-06-20
*/

#ifndef __INTEGRAL_SIFT_H__
#define __INTEGRAL_SIFT_H__

#include "IntegralImage.h"
#include "ScaleSpace.h"
#include <vector>

#define MAX_NUMBER_OF_FEATURES 1000
#define OUT_FILENAME "tmp.key"
#define FEATURE_DIMENSION 128
#define MY_EPS 1e-6

struct SiftFeature{
	FLOAT x, y, scale, orientation;
	char descriptor[FEATURE_DIMENSION];
	SiftFeature();
	SiftFeature(FLOAT _x, FLOAT _y, FLOAT _scale, FLOAT _orientation){
		x = _x; y = _y; scale = _scale; orientation = _orientation;
	}
};

class IntegralSift{
protected:
	// scalespace
	ScaleSpace* pSS;
	int maxT, dT;
	// order of the integral image
	int order;	

	// validity thresholds for potential feature to be valid
	FLOAT contrastThreshold;
	FLOAT hessianThreshold;

	// pointer to the original (integral) image
	IntegralImage* pIm;

	// feature descriptor dimension
	int descriptorDim;
	
	// list of detected siftFeatures
	std::vector<SiftFeature> features;

	// calculate sigma given T (and order of the B-spline)
	inline FLOAT calcSigma(int T) { return (FLOAT)sqrt((FLOAT)order/12.0)*T; }
	inline int calcMargin(int T) { return T/2*order; } //{ return (T+1)/2*order; }
	inline int calcT(int d) {return dT*d; }
	inline iType calcNormFactor(int T) { iType tmp = (iType)pow((FLOAT)mySquare(T), order); if (tmp > 0) return tmp; else return 1;}
//	inline int calcNormFactor(int T) { return (int)pow((float)mySquare(T), (int)order); }

public:
	/// default constructor
	IntegralSift();
	/// initialize by passing the image
	/* params:	im -- integral image
	*			maxT -- maximum scale (in T)
	*			dT -- step size for scale-space construction (in T)
	*/
	IntegralSift(IntegralImage* im, int _maxT, int _dT);

	/// destructor
	~IntegralSift();

	/// construct scale-space layer-by-layer
	void constructScaleSpace();
	/// find sift features
	void findSift();
	/// checks validity of a point to be a Sift feature (must pass the contrast + hessian test)
	bool isValid(int x, int y, int layer);

	// highest call to perform all SIFT procedure and save the result
	void compute();

	/// dumps the computed sift feature in the temporary file
	void dumpFeatures(char* filename);

};

#endif
